#ifndef _IKEV2G_PRF_PLUS_H_
#define _IKEV2G_PRF_PLUS_H_

typedef struct gost_prf_plus_t gost_prf_plus_t;

#include "blob.h"

/**
 * @brief Implementation of the prf+ based on Streebog-512.
 *
 * Any key handle can be used as key.
 * Can accumulate from bytes or key handles.
 */
struct gost_prf_plus_t {

    /**
     * @brief Get prf+ value.
     *
     * Saves pseudorandom value into @p buffer.
     *
     * @param this	Pointer to the gost_prf_plus_t structure.
     * @param length    Length in bytes of pseudorandom value.
     * @param buffer	Output buffer where hash value will be stored.
     * @return		True on success.
     */
    bool(*get_bytes)(gost_prf_plus_t *this, size_t length, uint8_t *buffer)
	WARN_UNUSED_RESULT;

    /**
     * @brief Get prf+ value.
     *
     * Allocates and saves pseudorandom value into @p blob.
     *
     * @param this	Pointer to the gost_prf_plus_t structure.
     * @param length    Length in bytes of pseudorandom value.
     * @param blob	Uninitialized or empty blob where hash will be stored.
     * @return		True on success.
     */
    bool(*allocate_bytes)(gost_prf_plus_t *this, size_t length, blob_t *blob)
	WARN_UNUSED_RESULT;

    /**
     * @brief Derive key.
     *
     * Derive key of specific length.
     *
     * @param this	Pointer to the gost_prf_plus_t structure.
     * @param bits      Length in bits of the derived key.
     * @return		Derived key handler on success or 0 otherwise.
     */
    uintptr_t (*derive_key)(gost_prf_plus_t *this, int bits);

    /**
     * @brief Add more data to the hash.
     *
     * Adds more seed data to the hash.
     *
     * @param this	Pointer to the gost_prf_plus_t structure.
     * @param seed	Seed data.
     * @return		True on success.
     */
    bool (*seed_bytes)(gost_prf_plus_t *this, blob_t seed);

    /**
     * @brief Add more key data to the hash.
     *
     * @param this	Pointer to the gost_prf_plus_t structure.
     * @param key	Key handler.
     * @return		True on success.
     */
    bool (*seed_key)(gost_prf_plus_t *this, uintptr_t key);

    /**
     * @brief Destructor of gost_prf_plus_t.
     *
     * @param this	Pointer to the ikev2_gost_t structure.
     */
    void (*destroy)(gost_prf_plus_t *this);
};

/**
 * @brief Create PRF+ from HMAC.
 *
 * @param prov		CSP context.
 * @param key		Key handle.
 * @return		Pointer to the gost_prf_plus_t structure on success.
 */
gost_prf_plus_t *gost_prf_plus_create(uintptr_t prov, uintptr_t key);

#endif /* _IKEV2G_PRF_PLUS_H_ */
