/*
 * LICENSE AND COPYRIGHT
 */

#ifndef _IKEV2G_BLOB_H_
#define _IKEV2G_BLOB_H_

#include <string.h>
#include <stdarg.h>
#include <stdint.h>
#ifdef HAVE_ALLOCA_H
#include <alloca.h>
#endif

#include "types.h"

#ifdef WIN32
#define MALLOC(size) LocalAlloc(LMEM_FIXED, (size))
#define FREE(p) LocalFree((p))
#else // WIN32
#define MALLOC(size) malloc((size))
#define FREE(p) free((p))
#endif

typedef struct blob_t blob_t;

/**
 * General purpose pointer/length struct.
 */
struct blob_t {
	/** Pointer to start of data */
	uint8_t *ptr;
	/** Length of data in bytes */
	size_t len;
};

/**
 * Create a new blob pointing to "ptr" with length "len"
 */
static inline blob_t blob_create(uint8_t *ptr, size_t len)
{
	//blob_t blob = {ptr, len};
	blob_t blob;
	blob.ptr = ptr;
	blob.len = len;
	return blob;
}

/**
 * Create a clone of a blob pointing to "ptr"
 */
blob_t blob_create_clone(uint8_t *ptr, blob_t blob);

/**
 * Calculate length of multiple blobs
 */
size_t blob_length(const char *mode, ...);

/**
 * Concatenate blobs into a blob pointing to "ptr".
 *
 * The mode string specifies the number of blobs, and how to handle each of
 * them with a single character: 'c' for copy (allocate new blob), 'm' for move
 * (free given blob) or 's' for sensitive-move (clear given blob, then free).
 */
blob_t blob_create_cat(uint8_t *ptr, const char* mode, ...);

/**
 * Split up a blob into parts, "mode" is a string of "a" (alloc),
 * "c" (copy) and "m" (move). Each letter say for the corresponding blob if
 * it should get allocated on heap, copied into existing blob, or the blob
 * should point into "blob". The length of each part is an argument before
 * each target blob. E.g.:
 * blob_split(blob, "mcac", 3, &a, 7, &b, 5, &c, d.len, &d);
 */
void blob_split(blob_t blob, const char *mode, ...);

/**
 * Convert a blob of data to hex encoding.
 *
 * The resulting string is '\\0' terminated, but the blob does not include
 * the '\\0'. If buf is supplied, it must hold at least (blob.len * 2 + 1).
 *
 * @param blob			data to convert to hex encoding
 * @param buf			buffer to write to, NULL to MALLOC
 * @param uppercase		TRUE to use uppercase letters
 * @return				blob of encoded data
 */
blob_t blob_to_hex(blob_t blob, char *buf, BOOL uppercase);

/**
 * Convert a hex encoded in a binary blob.
 *
 * If buf is supplied, it must hold at least (hex.len / 2) + (hex.len % 2)
 * bytes. It is filled by the right to give correct values for short inputs.
 *
 * @param hex			hex encoded input data
 * @param buf			buffer to write decoded data, NULL to MALLOC
 * @return				converted data
 */
blob_t blob_from_hex(blob_t hex, char *buf);

/**
 * Convert a blob of data to its base64 encoding.
 *
 * The resulting string is '\\0' terminated, but the blob does not include
 * the '\\0'. If buf is supplied, it must hold at least (blob.len * 4 / 3 + 1).
 *
 * @param blob			data to convert
 * @param buf			buffer to write to, NULL to MALLOC
 * @return				blob of encoded data
 */
blob_t blob_to_base64(blob_t blob, char *buf);

/**
 * Convert a base64 in a binary blob.
 *
 * If buf is supplied, it must hold at least (base64.len / 4 * 3).
 *
 * @param base64		base64 encoded input data
 * @param buf			buffer to write decoded data, NULL to MALLOC
 * @return				converted data
 */
blob_t blob_from_base64(blob_t base64, char *buf);

/**
 * Convert a blob of data to its base32 encoding.
 *
 * The resulting string is '\\0' terminated, but the blob does not include
 * the '\\0'. If buf is supplied, it must hold (blob.len * 8 / 5 + 1) bytes.
 *
 * @param blob			data to convert
 * @param buf			buffer to write to, NULL to MALLOC
 * @return				blob of encoded data
 */
blob_t blob_to_base32(blob_t blob, char *buf);

/**
 * Free contents of a blob
 */
static inline void blob_free(blob_t *blob)
{
	FREE(blob->ptr);
	memset(blob, 0, sizeof(blob_t));
}

/**
 * Overwrite the contents of a blob and free it
 */
static inline void blob_clear(blob_t *blob)
{
	if (blob->ptr)
	{
		memset(blob->ptr, 0, blob->len);
		blob_free(blob);
	}
}

/**
 * Initialize a blob using a char array
 */
#define blob_from_chars(...) ((blob_t){(uint8_t[]){__VA_ARGS__}, sizeof((uint8_t[]){__VA_ARGS__})})

/**
 * Initialize a blob to point to a thing
 */
#define blob_from_thing(thing) blob_create((uint8_t*)&(thing), sizeof(thing))

/**
 * Initialize a blob from a string, not containing 0-terminator
 */
#define blob_from_str(str) ({char *x = (str); blob_create((uint8_t*)x, strlen(x));})

/**
 * Allocate a blob on the heap
 */
#define blob_alloc(x) (blob_create((x) ? MALLOC((x)) : NULL, (x)))

/**
 * Allocate a blob on the stack
 */
#define blob_alloca(x) (blob_create((x) ? alloca((x)) : NULL, (x)))

/**
 * Clone a blob on heap
 */
#define blob_clone(blob) (blob_create_clone((blob).len ? MALLOC((blob).len) : NULL, (blob)))

/**
 * Clone a blob on stack
 */
#define blob_clonea(blob) (blob_create_clone((blob).len ? alloca((blob).len) : NULL, (blob)))

/**
 * Concatenate blobs into a blob on heap
 */
#define blob_cat(mode, ...) blob_create_cat(MALLOC(blob_length(mode, __VA_ARGS__)), mode, __VA_ARGS__)

/**
 * Concatenate blobs into a blob on stack
 */
#define blob_cata(mode, ...) blob_create_cat(alloca(blob_length(mode, __VA_ARGS__)), mode, __VA_ARGS__)

/**
 * Skip n bytes in blob (forward pointer, shorten length)
 */
static inline blob_t blob_skip(blob_t blob, size_t bytes)
{
	blob_t blob_empty = { 0 };
	if (blob.len > bytes)
	{
		blob.ptr += bytes;
		blob.len -= bytes;
		return blob;
	}
	return blob_empty;
}

/**
 * Skip any leading zero-valued bytes
 */
static inline blob_t blob_skip_zero(blob_t blob)
{
	while (blob.len > 1 && *blob.ptr == 0x00)
	{
		blob.ptr++;
		blob.len--;
	}
	return blob;
}

/**
 * Copy the data from src to dst, left-padded with chr if dst is longer,
 * otherwise data is copied truncated on the left.
 *
 * @param dst			data is copied here
 * @param src			data is copied from here
 * @param chr			value to use for padding if necessary
 * @return				the destination blob
 */
blob_t blob_copy_pad(blob_t dst, blob_t src, uint8_t chr);

/**
 *  Compare two blobs, returns zero if a equals b
 *  or negative/positive if a is small/greater than b
 */
int blob_compare(blob_t a, blob_t b);

/**
 * Compare two blobs for equality,
 * NULL blobs are never equal.
 */
static inline bool blob_equals(blob_t a, blob_t b)
{
	return a.ptr != NULL  && b.ptr != NULL &&
			a.len == b.len && !memcmp(a.ptr, b.ptr, a.len);
}

/**
 * Compare two blobs (given as pointers) for equality (useful as callback),
 * NULL blobs are never equal.
 */
static inline bool blob_equals_ptr(blob_t *a, blob_t *b)
{
	return a != NULL && b != NULL && blob_equals(*a, *b);
}

/**
 * Increment a blob, as it would represent a network order integer.
 *
 * @param blob			blob to increment
 * @return				TRUE if an overflow occurred
 */
bool blob_increment(blob_t blob);

/**
 * Check if a blob has printable characters only.
 *
 * If sane is given, blob is cloned into sane and all non printable characters
 * get replaced by "replace".
 *
 * @param blob			blob to check for printability
 * @param sane			pointer where sane version is allocated, or NULL
 * @param replace		character to use for replaceing unprintable characters
 * @return				TRUE if all characters in blob are printable
 */
bool blob_printable(blob_t blob, blob_t *sane, char replace);

/**
 * Seed initial key for blob_hash().
 *
 * This call should get invoked once during startup. This is usually done
 * by calling library_init(). Calling it multiple times is safe, it gets
 * executed just once.
 */
void blob_hash_seed(void);

/**
 * Computes a 32 bit hash of the given blob.
 *
 * @note The output of this function is randomized, that is, it will only
 * produce the same output for the same input when calling it from the same
 * process.  For a more predictable hash function use blob_hash_static()
 * instead.
 *
 * @note This hash is only intended for hash tables not for cryptographic
 * purposes.
 *
 * @param blob			data to hash
 * @return				hash value
 */
uint32_t blob_hash(blob_t blob);

/**
 * Incremental version of blob_hash. Use this to hash two or more blobs.
 *
 * @param blob			data to hash
 * @param hash			previous hash value
 * @return				hash value
 */
uint32_t blob_hash_inc(blob_t blob, uint32_t hash);

/**
 * Computes a 32 bit hash of the given blob.
 *
 * Compared to blob_hash() this will always calculate the same output for the
 * same input.  Therefore, it should not be used for hash tables (to prevent
 * hash flooding).
 *
 * @note This hash is not intended for cryptographic purposes.
 *
 * @param blob			data to hash
 * @return				hash value
 */
uint32_t blob_hash_static(blob_t blob);

/**
 * Incremental version of blob_hash_static(). Use this to hash two or more
 * blobs in a predictable way.
 *
 * @param blob			data to hash
 * @param hash			previous hash value
 * @return				hash value
 */
uint32_t blob_hash_static_inc(blob_t blob, uint32_t hash);

/**
 * Computes a quick MAC from the given blob and key using SipHash.
 *
 * The key must have a length of 128-bit (16 bytes).
 *
 * @note While SipHash has strong features using it for cryptographic purposes
 * is not recommended (in particular because of the rather short output size).
 *
 * @param blob			data to process
 * @param key			key to use
 * @return				MAC for given input and key
 */
uint64_t blob_mac(blob_t blob, uint8_t *key);

/**
 * Calculate the Internet Checksum according to RFC 1071 for the given blob.
 *
 * If the result is used with blob_internet_checksum_inc() and the data length
 * is not a multiple of 16 bit the checksum bytes have to be swapped to
 * compensate the even/odd alignment.
 *
 * @param data			data to process
 * @return				checksum (one's complement, network order)
 */
uint16_t blob_internet_checksum(blob_t data);

/**
 * Extend the given Internet Checksum (one's complement, in network byte order)
 * with the given data.
 *
 * If data is not a multiple of 16 bits the checksum may have to be swapped to
 * compensate even/odd alignment (see blob_internet_checksum()).
 *
 * @param data			data to process
 * @param checksum		previous checksum (one's complement, network order)
 * @return				checksum (one's complement, network order)
 */
uint16_t blob_internet_checksum_inc(blob_t data, uint16_t checksum);

#endif /* _IKEV2G_BLOB_H_ */
