/*
 * Copyright (C) 1997-2003 by Objective Systems, Inc.
 *
 * This software is furnished under a license and may be used and copied
 * only in accordance with the terms of such license and with the
 * inclusion of the above copyright notice. This software or any other
 * copies thereof may not be provided or otherwise made available to any
 * other person. No title to and ownership of the software is hereby
 * transferred.
 *
 * The information in this software is subject to change without notice
 * and should not be construed as a commitment by Objective Systems, Inc.
 *
 * PROPRIETARY NOTICE
 *
 * This software is an unpublished work subject to a confidentiality agreement
 * and is protected by copyright and trade secret law.  Unauthorized copying,
 * redistribution or other use of this work is prohibited.
 *
 * The above notice of copyright on this source code product does not indicate
 * any actual or intended publication of such source code.
 *
 *****************************************************************************/

/** 
 * @file ASN1BERInputStream.h 
 * The C++ definitions for ASN.1 BER input streams.
 */

#ifndef _ASN1BERINPUTSTREAM_H_
#define _ASN1BERINPUTSTREAM_H_

#include "asn1berStream.h"
#include "ASN1Stream.h"
#include "OSCSocket.h"

/** @defgroup iberstrmclas C++ classes for streaming BER decoding.
 * @ingroup bercppruntime 
 * These classes are used to perform BER decoding directly from a stream (file,
 * network, memory). @{
 */

/** @defgroup berstrmdecwrappers Wrapper classes for BER decoding by using operator >>
 * These wrapper classes can be used for decoding with using the >> operator.
 * For example: ASN1BERInputStream is; ..... is >> OSDecStrmTag (v1) >>
 * OSDecStrmLength (v2) >> OSDecStrmNull (); @{
 */

/**
 * A wrapper class for decoding a tag value.
 */
struct EXTERNBER OSDecStrmTag {
   ASN1TAG& value; ///< a value

   /** A constructor */
   OSDecStrmTag (ASN1TAG& val) : value (val) { }
   /** A constructor */
   OSDecStrmTag (OSDecStrmTag& val) : value (val.value) { }
} ;

/**
 * A wrapper class for decoding a length value.
 */
struct EXTERNBER OSDecStrmLength {
   ASN1INT& value; ///< a value

   /** A constructor */
   OSDecStrmLength (ASN1INT& val) : value (val) { }
   /** A constructor */
   OSDecStrmLength (OSDecStrmLength& val) : value (val.value) { }
} ;

/**
 * A wrapper class for decoding a tag and length values.
 */
struct EXTERNBER OSDecStrmTagAndLen {
   ASN1TAG& tag; ///< a tag
   ASN1INT& len; ///< a length

   /** A constructor */
   OSDecStrmTagAndLen (ASN1TAG& _tag, ASN1INT& _len) : tag (_tag), len (_len) {}
   /** A constructor */
   OSDecStrmTagAndLen (OSDecStrmTagAndLen& val) : 
      tag (val.tag), len (val.len) { }
} ;

/**
 * A wrapper class for decoding an end-of-contents octets.
 */
struct EXTERNBER OSDecStrmEoc {
   /** A constructor */
   OSDecStrmEoc () {}
} ;

#define OSDECSTRMEOC OSDecStrmEoc()

/**
 * A wrapper class for decoding an implicit integer value.
 */
struct EXTERNBER OSDecStrmImplInt : public OSDecStrmInt {
   ASN1INT len; ///< a length

   /** A constructor */
   OSDecStrmImplInt (ASN1INT& val, ASN1INT _len) : 
      OSDecStrmInt (val), len (_len) { }
} ;

/**
 * A wrapper class for decoding an implicit 8-bit integer value.
 */
struct EXTERNBER OSDecStrmImplInt8 : public OSDecStrmInt8 {
   ASN1INT len; ///< a length

   /** A constructor */
   OSDecStrmImplInt8 (ASN1INT8& val, ASN1INT _len) : 
      OSDecStrmInt8 (val), len (_len) { }
} ;

/**
 * A wrapper class for decoding an implicit 16-bit integer value.
 */
struct EXTERNBER OSDecStrmImplInt16 : public OSDecStrmInt16 {
   ASN1INT len; ///< a length

   /** A constructor */
   OSDecStrmImplInt16 (ASN1SINT& val, ASN1INT _len) : 
      OSDecStrmInt16 (val), len (_len) { }
} ;

/**
 * A wrapper class for decoding an implicit 64-bit integer value.
 */
struct EXTERNBER OSDecStrmImplInt64 : public OSDecStrmInt64 {
   ASN1INT len; ///< a length

   /** A constructor */
   OSDecStrmImplInt64 (ASN1INT64& val, ASN1INT _len) : 
      OSDecStrmInt64 (val), len (_len) { }
} ;

/**
 * A wrapper class for decoding an implicit unsigned integer value.
 */
struct EXTERNBER OSDecStrmImplUInt : public OSDecStrmUInt {
   ASN1INT len; ///< a length

   /** A constructor */
   OSDecStrmImplUInt (ASN1UINT& val, ASN1INT _len) : 
      OSDecStrmUInt (val), len (_len) { }
} ;

/**
 * A wrapper class for decoding an implicit 8-bit unsigned integer value.
 */
struct EXTERNBER OSDecStrmImplUInt8 : public OSDecStrmUInt8 {
   ASN1INT len; ///< a length

   /** A constructor */
   OSDecStrmImplUInt8 (ASN1UINT8& val, ASN1INT _len) : 
      OSDecStrmUInt8 (val), len (_len) { }
} ;

/**
 * A wrapper class for decoding an implicit 16-bit unsigned integer value.
 */
struct EXTERNBER OSDecStrmImplUInt16 : public OSDecStrmUInt16 {
   ASN1INT len; ///< a length

   /** A constructor */
   OSDecStrmImplUInt16 (ASN1USINT& val, ASN1INT _len) : 
      OSDecStrmUInt16 (val), len (_len) { }
} ;

/**
 * A wrapper class for decoding an implicit 64-bit unsigned integer value.
 */
struct EXTERNBER OSDecStrmImplUInt64 : public OSDecStrmUInt64 {
   ASN1INT len; ///< a length

   /** A constructor */
   OSDecStrmImplUInt64 (ASN1UINT64& val, ASN1INT _len) : 
      OSDecStrmUInt64 (val), len (_len) { }
} ;

/**
 * A wrapper class for decoding an implicit ENUMERATED value.
 */
struct EXTERNBER OSDecStrmImplEnum : public OSDecStrmEnum {
   ASN1INT len; ///< a length

   /** A constructor */
   OSDecStrmImplEnum (ASN1ENUM& val, ASN1INT _len) : 
      OSDecStrmEnum (val), len (_len) { }
} ;

/**
 * A wrapper class for decoding an implicit big integer value.
 */
struct EXTERNBER OSDecStrmImplBigInt : public OSDecStrmBigInt {
   ASN1INT len; ///< a length

   /** A constructor */
   OSDecStrmImplBigInt (const char*& pval, ASN1INT _len) : 
      OSDecStrmBigInt (pval), len (_len) { }
} ;

/**
 * A wrapper class for decoding an implicit REAL value.
 */
struct EXTERNBER OSDecStrmImplReal : public OSDecStrmReal {
   ASN1INT len; ///< a length

   /** A constructor */
   OSDecStrmImplReal (ASN1REAL& val, ASN1INT _len) : 
      OSDecStrmReal (val), len (_len) { }
} ;

/**
 * A wrapper class for decoding an implicit sized BIT STRING value.
 */
struct EXTERNBER OSDecStrmImplBitStr : public OSDecStrmBitStr {
   ASN1INT len; ///< a length

   /** A constructor */
   OSDecStrmImplBitStr (ASN1UINT _numbits, ASN1OCTET*& _data, ASN1INT _len) : 
      OSDecStrmBitStr (_numbits, _data), len (_len) { }
} ;

/**
 * A wrapper class for decoding an implicit dynamic BIT STRING value.
 */
struct EXTERNBER OSDecStrmImplDynBitStr : public OSDecStrmDynBitStr {
   ASN1INT len; ///< a length

   /** A constructor */
   OSDecStrmImplDynBitStr (ASN1DynBitStr& _bs, ASN1INT _len) : 
      OSDecStrmDynBitStr (_bs), len (_len) { }
} ;

/**
 * A wrapper class for decoding an implicit sized OCTET STRING value.
 */
struct EXTERNBER OSDecStrmImplOctStr : public OSDecStrmOctStr {
   ASN1INT len; ///< a length

   /** A constructor */
   OSDecStrmImplOctStr (ASN1UINT _numocts, ASN1OCTET*& _data, ASN1INT _len) :
      OSDecStrmOctStr (_numocts, _data), len (_len) { }
} ;

/**
 * A wrapper class for decoding an implicit dynamic OCTET STRING value.
 */
struct EXTERNBER OSDecStrmImplDynOctStr : public OSDecStrmDynOctStr {
   ASN1INT len; ///< a length

   /** A constructor */
   OSDecStrmImplDynOctStr (ASN1DynOctStr& _os, ASN1INT _len) : 
      OSDecStrmDynOctStr (_os), len (_len) { }

} ;

/**
 * A wrapper class for decoding an implicit BOOLEAN value.
 */
struct EXTERNBER OSDecStrmImplBool : public OSDecStrmBool {
   ASN1INT len; ///< a length

   /** A constructor */
   OSDecStrmImplBool (ASN1BOOL& val, ASN1INT _len) : 
      OSDecStrmBool (val), len (_len) { }
} ;

/**
 * A wrapper class for decoding an implicit character string value.
 */
struct EXTERNBER OSDecStrmImplCharStr : public OSDecStrmCharStr {
   ASN1INT len; ///< a length

   /** A constructor */
   OSDecStrmImplCharStr (const char*& pval, ASN1TAG tag, ASN1INT _len) : 
      OSDecStrmCharStr (pval, tag), len (_len) { }
} ;

/**
 * A wrapper class for decoding an implicit OBJECT IDENTIFIER value.
 */
struct EXTERNBER OSDecStrmImplObjId : public OSDecStrmObjId {
   ASN1INT len; ///< a length

   /** A constructor */
   OSDecStrmImplObjId (ASN1OBJID& oid, ASN1INT _len) : 
      OSDecStrmObjId (oid), len (_len) { }
} ;

/**
 * A wrapper class for decoding an implicit RELATIVE-OID value.
 */
struct EXTERNBER OSDecStrmImplRelativeOID : public OSDecStrmObjId {
   ASN1INT len; ///< a length

   /** A constructor */
   OSDecStrmImplRelativeOID (ASN1OBJID& oid, ASN1INT _len) : 
      OSDecStrmObjId (oid), len (_len) { }
} ;

/**
 * A wrapper class for decoding an implicit NULL value.
 */
struct EXTERNBER OSDecStrmImplNull {};

/**
 * A wrapper class for decoding an implicit BMPString value.
 */
struct EXTERNBER OSDecStrmImplBMPString : public OSDecStrmBMPString {
   ASN1INT len; ///< a length

   /** A constructor */
   OSDecStrmImplBMPString (Asn116BitCharString& val, ASN1INT _len) :
      OSDecStrmBMPString (val), len (_len) { }
} ;

/**
 * A wrapper class for decoding an implicit UniversalString value.
 */
struct EXTERNBER OSDecStrmImplUnivString : public OSDecStrmUnivString {
   ASN1INT len; ///< a length

   /** A constructor */
   OSDecStrmImplUnivString (Asn132BitCharString& val, ASN1INT _len) :
      OSDecStrmUnivString (val), len (_len) { }
} ;

/** @} */

/**
 * This class is a base class for other ASN.1 BER input stream's classes. It is
 * derived from the ASN1Stream base class. It contains variables and methods
 * specific to streaming decoding of BER messages. It is used to manage the
 * input stream containing the ASN.1 message to be decoded.
 */
class EXTERNBER ASN1BERInputStream : public ASN1Stream {
 protected:
   /**
    * A default constructor.
    *
    * @exception OSCStreamException    stream can't be created or initialized.
    */
   ASN1BERInputStream () { }
 public:
   /**
    * Decodes an ASN.1 constructed object from the stream.
    *
    * @param val          A reference to an object to be decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (ASN1CType& val); 

   /**
    * Decodes a tag value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmTag& val); 

   /**
    * Decodes a length determinant from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmLength& val); 

   /**
    * Decodes a tag value and length from the stream.
    *
    * @param val          A reference to a wrapper class with values to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmTagAndLen& val); 

   /**
    * Decodes a end-of-contents octets from the stream.
    *
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmEoc&); 

   /**
    * Decodes an integer value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmInt& val); 

   /**
    * Decodes an 8-bit integer value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmInt8& val); 

   /**
    * Decodes a 16-bit integer value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmInt16& val); 

   /**
    * Decodes a 64-bit integer value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmInt64& val); 

   /**
    * Decodes an unsigned integer value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmUInt& val); 

   /**
    * Decodes an 8-bit unsigned integer value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmUInt8& val); 

   /**
    * Decodes a 16-bit unsigned integer value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmUInt16& val); 

   /**
    * Decodes a 64-bit unsigned integer value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmUInt64& val); 

   /**
    * Decodes a big integer value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmBigInt& val); 

   /**
    * Decodes an ENUMERATED value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmEnum& val); 

   /**
    * Decodes a BOOLEAN value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmBool& val); 

   /**
    * Decodes a REAL value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmReal& val); 

   /**
    * Decodes a sized BIT STRING value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmBitStr& val); 

   /**
    * Decodes a sized OCTET STRING value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmOctStr& val); 

   /**
    * Decodes a dynamic BIT STRING value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmDynBitStr& val); 

   /**
    * Decodes a dynamic OCTET STRING value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmDynOctStr& val); 

   /**
    * Decodes a character string value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmCharStr& val); 

   /**
    * Decodes an OBJECT IDENTIFIER value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmObjId& val); 

   /**
    * Decodes a RELATIVE-OID value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmRelativeOID& val); 

   /**
    * Decodes a NULL value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmNull& val); 

   /**
    * Decodes a BMPString value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmBMPString& val); 

   /**
    * Decodes an UniversalString value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmUnivString& val); 

   // Implicit operators
   /**
    * Decodes an implicit integer value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmImplInt& val); 

   /**
    * Decodes an implicit 8-bit integer value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmImplInt8& val); 

   /**
    * Decodes an implicit 16-bit integer value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmImplInt16& val); 

   /**
    * Decodes an implicit 64-bit integer value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmImplInt64& val); 

   /**
    * Decodes an implicit unsigned integer value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmImplUInt& val); 

   /**
    * Decodes an implicit 8-bit unsigned integer value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmImplUInt8& val); 

   /**
    * Decodes an implicit 16-bit unsigned integer value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmImplUInt16& val); 

   /**
    * Decodes an implicit 64-bit unsigned integer value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmImplUInt64& val); 

   /**
    * Decodes an implicit big integer value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmImplBigInt& val); 

   /**
    * Decodes an implicit ENUMERATED value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmImplEnum& val); 

   /**
    * Decodes an implicit BOOLEAN value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmImplBool& val); 

   /**
    * Decodes an implicit REAL value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmImplReal& val); 

   /**
    * Decodes an implicit sized BIT STRING value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmImplBitStr& val); 

   /**
    * Decodes an implicit sized OCTET STRING value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmImplOctStr& val); 

   /**
    * Decodes an implicit dynamic BIT STRING value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmImplDynBitStr& val); 

   /**
    * Decodes an implicit dynamic OCTET STRING value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmImplDynOctStr& val); 

   /**
    * Decodes an implicit character string value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmImplCharStr& val); 

   /**
    * Decodes an implicit OBJECT IDENTIFIER value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmImplObjId& val); 

   /**
    * Decodes an implicit RELATIVE-OID value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmImplRelativeOID& val); 

   /**
    * Decodes an implicit NULL value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmImplNull& val); 

   /**
    * Decodes an implicit BMPString value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmImplBMPString& val); 

   /**
    * Decodes an implicit UniversalString value from the stream.
    *
    * @param val          A reference to a wrapper class with a value to be
    *                       decoded.
    * @return             reference to this class to perform sequential
    *                       decoding.
    */
   ASN1BERInputStream& operator >> (OSDecStrmImplUnivString& val); 

   /**
    * This method returns the total number of octets (bytes) already decoded
    * from the stream.
    *
    * @return             Number of octets (bytes) already decoded from the
    *                       stream.
    */
   int byteIndex ();

   /**
    * This method determines if the decoder has reached the end of a message
    * context block. This method could be called when decoding a SET or
    * SEQUENCE OF/SET OF construct.
    *
    * @param ccb          Reference to a 'context control block' structure.
    *                       This is basically a loop control mechanism to keep
    *                       the variable associated with parsing a nested
    *                       constructed element straight.
    * @return             Boolean value indicating whether or not the
    *                       end-of-context has been reached.
    */
   ASN1BOOL chkend (ASN1CCB& ccb);

   /**
    * This method decodes a variable of the ASN.1 INTEGER type. In this case,
    * the integer is assumed to be of a larger size than can fit in a C or C++
    * long type (normally 32 or 64 bits).
    *
    * @param pval                      Reference to a pointer to a variable to
    *                                    receive a decoded big integer value.
    * @param tagging                   An enumerated type whose value is set to
    *                                    either 'ASN1EXPL' (for explicit
    *                                    tagging) or 'ASN1IMPL' (for implicit).
    *                                    Controls whether the universal tag
    *                                    value for this type is added or not.
    *                                    Users will generally always set this
    *                                    value to 'ASN1EXPL'.
    * @param length                    The length, in octets, of the contents
    *                                    field to be decoded. This parameter
    *                                    only has meaning if the tagging
    *                                    parameter specifies implicit decoding.
    *                                    If explicit, the length is obtained
    *                                    from the decoded length field.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmBigInt
    */
   void decodeBigInt (const char*& pval, ASN1TagType tagging = ASN1EXPL, 
                      int length = 0);

   /**
    * This method decodes a variable of the ASN.1 BIT STRING type into a static
    * memory structure. It is used to decode a sized bit string production.
    *
    * @param pbits                     Pointer to a variable to receive the
    *                                    decoded bit string. This is assumed to
    *                                    be a static array large enough to hold
    *                                    the number of bits specified in the
    *                                    numbits input parameter.
    * @param numbits                   As input parameter it is a reference to
    *                                    an integer variable containing the
    *                                    size (in bits) of the sized ASN.1 bit
    *                                    string. An error will occur if the
    *                                    number of bits in the decoded string
    *                                    is larger than this value. Note that
    *                                    this is also used as an output
    *                                    variable - the actual number of
    *                                    decoded bits will be returned in this
    *                                    variable.
    * @param tagging                   An enumerated type whose value is set to
    *                                    either 'ASN1EXPL' (for explicit
    *                                    tagging) or 'ASN1IMPL' (for implicit).
    *                                    Controls whether the universal tag
    *                                    value for this type is added or not.
    *                                    Users will generally always set this
    *                                    value to 'ASN1EXPL'.
    * @param length                    The length, in octets, of the contents
    *                                    field to be decoded. This parameter
    *                                    only has meaning if the tagging
    *                                    parameter specifies implicit decoding.
    *                                    If explicit, the length is obtained
    *                                    from the decoded length field.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmBitStr
    */
   void decodeBitStr (ASN1OCTET* pbits, ASN1UINT& numbits, 
                      ASN1TagType tagging = ASN1EXPL, int length = 0);

   /**
    * This method decodes a variable of the ASN.1 BIT STRING type. It will
    * allocate dynamic memory to store the decoded result.
    *
    * @param val                       Reference to an ASN1DynBitStr variable
    *                                    to receive the decoded bit string.
    * @param tagging                   An enumerated type whose value is set to
    *                                    either 'ASN1EXPL' (for explicit
    *                                    tagging) or 'ASN1IMPL' (for implicit).
    *                                    Controls whether the universal tag
    *                                    value for this type is added or not.
    *                                    Users will generally always set this
    *                                    value to 'ASN1EXPL'.
    * @param length                    The length, in octets, of the contents
    *                                    field to be decoded. This parameter
    *                                    only has meaning if the tagging
    *                                    parameter specifies implicit decoding.
    *                                    If explicit, the length is obtained
    *                                    from the decoded length field.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmDynBitStr
    */
   void decodeBitStr (ASN1DynBitStr& val, ASN1TagType tagging = ASN1EXPL, 
                      int length = 0);

   /**
    * This method decodes a variable of the ASN.1 BMPString type.
    *
    * @param val                       Reference to a variable to receive the
    *                                    decoded BMPString value.
    * @param tagging                   An enumerated type whose value is set to
    *                                    either 'ASN1EXPL' (for explicit
    *                                    tagging) or 'ASN1IMPL' (for implicit).
    *                                    Controls whether the universal tag
    *                                    value for this type is added or not.
    *                                    Users will generally always set this
    *                                    value to 'ASN1EXPL'.
    * @param length                    The length, in octets, of the contents
    *                                    field to be decoded. This parameter
    *                                    only has meaning if the tagging
    *                                    parameter specifies implicit decoding.
    *                                    If explicit, the length is obtained
    *                                    from the decoded length field.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmBMPStr
    */
   void decodeBMPStr (Asn116BitCharString& val, ASN1TagType tagging = ASN1EXPL, 
                      int length = 0);

   /**
    * This method decodes a variable of the ASN.1 BOOLEAN type.
    *
    * @param val                       Reference to a variable to receive the
    *                                    decoded boolean value.
    * @param tagging                   An enumerated type whose value is set to
    *                                    either 'ASN1EXPL' (for explicit
    *                                    tagging) or 'ASN1IMPL' (for implicit).
    *                                    Controls whether the universal tag
    *                                    value for this type is added or not.
    *                                    Users will generally always set this
    *                                    value to 'ASN1EXPL'.
    * @param length                    The length, in octets, of the contents
    *                                    field to be decoded. This parameter
    *                                    only has meaning if the tagging
    *                                    parameter specifies implicit decoding.
    *                                    If explicit, the length is obtained
    *                                    from the decoded length field.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmBool
    */
   void decodeBool (ASN1BOOL& val, ASN1TagType tagging = ASN1EXPL, 
                    int length = 0);

   /**
    * This method decodes a variable of one of the ASN.1 8-bit character string
    * types. These types include IA5String, VisibleString, PrintableString, and
    * NumericString.
    *
    * @param pval                      Reference to a character string pointer
    *                                    variable to receive the decoded
    *                                    string. The string is stored as a
    *                                    standard null-terminated C string.
    *                                    Memory is allocated for the string by
    *                                    the ::rtMemAlloc function.
    * @param tagging                   An enumerated type whose value is set to
    *                                    either 'ASN1EXPL' (for explicit
    *                                    tagging) or 'ASN1IMPL' (for implicit).
    *                                    Controls whether the universal tag
    *                                    value for this type is added or not.
    *                                    Users will generally always set this
    *                                    value to 'ASN1EXPL'.
    * @param tag                       The ASN.1 tag to be decoded. This
    *                                    parameter is passed using the ASN1C
    *                                    internal tag representation. It is
    *                                    passed as an unsigned 32-bit integer.
    *                                    This parameter only has meaning if the
    *                                    tagging parameter specifies explicit
    *                                    decoding.
    * @param length                    The length, in octets, of the contents
    *                                    field to be decoded. This parameter
    *                                    only has meaning if the tagging
    *                                    parameter specifies implicit decoding.
    *                                    If explicit, the length is obtained
    *                                    from the decoded length field.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmCharStr
    */
   void decodeCharStr (const char*& pval, ASN1TagType tagging = ASN1EXPL, 
                       ASN1TAG tag = 0, int length = 0);

   /**
    * This method decodes a variable of the ASN.1 ENUMERATED type.
    *
    * @param val                       Reference to a variable to receive the
    *                                    decoded enumerated value.
    * @param tagging                   An enumerated type whose value is set to
    *                                    either 'ASN1EXPL' (for explicit
    *                                    tagging) or 'ASN1IMPL' (for implicit).
    *                                    Controls whether the universal tag
    *                                    value for this type is added or not.
    *                                    Users will generally always set this
    *                                    value to 'ASN1EXPL'.
    * @param length                    The length, in octets, of the contents
    *                                    field to be decoded. This parameter
    *                                    only has meaning if the tagging
    *                                    parameter specifies implicit decoding.
    *                                    If explicit, the length is obtained
    *                                    from the decoded length field.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmEnum
    */
   void decodeEnum (ASN1ENUM& val, ASN1TagType tagging = ASN1EXPL, 
                    int length = 0);

   /**
    * This method decodes the end-of-contents octets.
    *
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmMatchEOC
    */
   void decodeEoc ();

   /**
    * This method decodes a variable of the ASN.1 INTEGER type.
    *
    * @param val                       Reference to a variable to receive a
    *                                    decoded 32-bit integer value.
    * @param tagging                   An enumerated type whose value is set to
    *                                    either 'ASN1EXPL' (for explicit
    *                                    tagging) or 'ASN1IMPL' (for implicit).
    *                                    Controls whether the universal tag
    *                                    value for this type is added or not.
    *                                    Users will generally always set this
    *                                    value to 'ASN1EXPL'.
    * @param length                    The length, in octets, of the contents
    *                                    field to be decoded. This parameter
    *                                    only has meaning if the tagging
    *                                    parameter specifies implicit decoding.
    *                                    If explicit, the length is obtained
    *                                    from the decoded length field.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmInt
    */
   void decodeInt (ASN1INT& val, ASN1TagType tagging = ASN1EXPL, 
                   int length = 0);

   /**
    * This method decodes an 8-bit variable of the ASN.1 INTEGER type.
    *
    * @param val                       Reference to a variable to receive a
    *                                    decoded 8-bit integer value.
    * @param tagging                   An enumerated type whose value is set to
    *                                    either 'ASN1EXPL' (for explicit
    *                                    tagging) or 'ASN1IMPL' (for implicit).
    *                                    Controls whether the universal tag
    *                                    value for this type is added or not.
    *                                    Users will generally always set this
    *                                    value to 'ASN1EXPL'.
    * @param length                    The length, in octets, of the contents
    *                                    field to be decoded. This parameter
    *                                    only has meaning if the tagging
    *                                    parameter specifies implicit decoding.
    *                                    If explicit, the length is obtained
    *                                    from the decoded length field.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmInt8
    */
   void decodeInt8 (ASN1INT8& val, ASN1TagType tagging = ASN1EXPL, 
                    int length = 0);

   /**
    * This method decodes a 16-bit variable of the ASN.1 INTEGER type.
    *
    * @param val                       Reference to a variable to receive a
    *                                    decoded 16-bit integer value.
    * @param tagging                   An enumerated type whose value is set to
    *                                    either 'ASN1EXPL' (for explicit
    *                                    tagging) or 'ASN1IMPL' (for implicit).
    *                                    Controls whether the universal tag
    *                                    value for this type is added or not.
    *                                    Users will generally always set this
    *                                    value to 'ASN1EXPL'.
    * @param length                    The length, in octets, of the contents
    *                                    field to be decoded. This parameter
    *                                    only has meaning if the tagging
    *                                    parameter specifies implicit decoding.
    *                                    If explicit, the length is obtained
    *                                    from the decoded length field.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmInt16
    */
   void decodeInt16 (ASN1SINT& val, ASN1TagType tagging = ASN1EXPL, 
                     int length = 0);

   /**
    * This method decodes a 64-bit variable of the ASN.1 INTEGER type.
    *
    * @param val                       Reference to a variable to receive a
    *                                    decoded 64-bit integer value.
    * @param tagging                   An enumerated type whose value is set to
    *                                    either 'ASN1EXPL' (for explicit
    *                                    tagging) or 'ASN1IMPL' (for implicit).
    *                                    Controls whether the universal tag
    *                                    value for this type is added or not.
    *                                    Users will generally always set this
    *                                    value to 'ASN1EXPL'.
    * @param length                    The length, in octets, of the contents
    *                                    field to be decoded. This parameter
    *                                    only has meaning if the tagging
    *                                    parameter specifies implicit decoding.
    *                                    If explicit, the length is obtained
    *                                    from the decoded length field.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmInt64
    */
   void decodeInt64 (ASN1INT64& val, ASN1TagType tagging = ASN1EXPL, 
                     int length = 0);

   /**
    * This method decodes a BER length determinant value.
    *
    * @param length                    Reference to a variable to receive the
    *                                    decoded length of the tagged
    *                                    component. The returned value will
    *                                    either be the actual length or the
    *                                    special constant 'ASN_K_INDEFLEN',
    *                                    which indicates indefinite length.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmLength
    */
   void decodeLength (ASN1INT& length);
   
   /**
    * This method decodes a variable of the ASN.1 NULL type.
    *
    * @param tagging                   An enumerated type whose value is set to
    *                                    either 'ASN1EXPL' (for explicit
    *                                    tagging) or 'ASN1IMPL' (for implicit).
    *                                    Controls whether the universal tag
    *                                    value for this type is added or not.
    *                                    Users will generally always set this
    *                                    value to 'ASN1EXPL'.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmNull
    */
   void decodeNull (ASN1TagType tagging = ASN1EXPL);

   /**
    * This method decodes an ASN.1 constructed object from the stream.
    *
    * @param val                       A reference to an object to be decoded.
    * @exception OSCStreamException    if error occurred.
    */
   void decodeObj (ASN1CType& val);

   /**
    * This method decodes a variable of the ASN.1 OBJECT IDENTIFIER type.
    *
    * @param val                       Reference to a variable to receive the
    *                                    decoded OBJECT IDENTIFIER value.
    * @param tagging                   An enumerated type whose value is set to
    *                                    either 'ASN1EXPL' (for explicit
    *                                    tagging) or 'ASN1IMPL' (for implicit).
    *                                    Controls whether the universal tag
    *                                    value for this type is added or not.
    *                                    Users will generally always set this
    *                                    value to 'ASN1EXPL'.
    * @param length                    The length, in octets, of the contents
    *                                    field to be decoded. This parameter
    *                                    only has meaning if the tagging
    *                                    parameter specifies implicit decoding.
    *                                    If explicit, the length is obtained
    *                                    from the decoded length field.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmObjId
    */
   void decodeObjId (ASN1OBJID& val, ASN1TagType tagging = ASN1EXPL, 
                     int length = 0);

   /**
    * This method decodes a variable of the ASN.1 OBJECT IDENTIFIER type using
    * 64-bit subidentifiers..
    *
    * @param val                       Reference to a variable to receive the
    *                                    decoded 64-bit OBJECT IDENTIFIER
    *                                    value.
    * @param tagging                   An enumerated type whose value is set to
    *                                    either 'ASN1EXPL' (for explicit
    *                                    tagging) or 'ASN1IMPL' (for implicit).
    *                                    Controls whether the universal tag
    *                                    value for this type is added or not.
    *                                    Users will generally always set this
    *                                    value to 'ASN1EXPL'.
    * @param length                    The length, in octets, of the contents
    *                                    field to be decoded. This parameter
    *                                    only has meaning if the tagging
    *                                    parameter specifies implicit decoding.
    *                                    If explicit, the length is obtained
    *                                    from the decoded length field.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmObjId64
    */
   void decodeObjId64 (ASN1OID64& val, ASN1TagType tagging = ASN1EXPL, 
                       int length = 0);

   /**
    * This method decodes a variable of the ASN.1 OCTET STRING type into a
    * static memory structure. It is used to decode a sized octet string
    * production.
    *
    * @param pocts                     Pointer to a variable to receive the
    *                                    decoded octet string. This is assumed
    *                                    to be a static array large enough to
    *                                    hold the number of octets specified in
    *                                    the numocts input parameter.
    * @param numocts                   Reference to an integer variable
    *                                    containing the size (in octets) of the
    *                                    sized ASN.1 octet string. An error
    *                                    will occur if the number of octets in
    *                                    the decoded string is larger than this
    *                                    value. Note that this is also used as
    *                                    an output variable - the actual number
    *                                    of decoded octets will be returned in
    *                                    this variable.
    * @param tagging                   An enumerated type whose value is set to
    *                                    either 'ASN1EXPL' (for explicit
    *                                    tagging) or 'ASN1IMPL' (for implicit).
    *                                    Controls whether the universal tag
    *                                    value for this type is added or not.
    *                                    Users will generally always set this
    *                                    value to 'ASN1EXPL'.
    * @param length                    The length, in octets, of the contents
    *                                    field to be decoded. This parameter
    *                                    only has meaning if the tagging
    *                                    parameter specifies implicit decoding.
    *                                    If explicit, the length is obtained
    *                                    from the decoded length field.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmOctStr
    */
   void decodeOctStr (ASN1OCTET* pocts, ASN1UINT& numocts, 
                      ASN1TagType tagging = ASN1EXPL, int length = 0);

   /**
    * This method decodes a variable of the ASN.1 OCTET STRING type. It will
    * allocate dynamic memory to store the decoded result.
    *
    * @param val                       Reference to an ASN1DynOctStr variable
    *                                    to receive the decoded octet string.
    * @param tagging                   An enumerated type whose value is set to
    *                                    either 'ASN1EXPL' (for explicit
    *                                    tagging) or 'ASN1IMPL' (for implicit).
    *                                    Controls whether the universal tag
    *                                    value for this type is added or not.
    *                                    Users will generally always set this
    *                                    value to 'ASN1EXPL'.
    * @param length                    The length, in octets, of the contents
    *                                    field to be decoded. This parameter
    *                                    only has meaning if the tagging
    *                                    parameter specifies implicit decoding.
    *                                    If explicit, the length is obtained
    *                                    from the decoded length field.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmDynOctStr
    */
   void decodeOctStr (ASN1DynOctStr& val, ASN1TagType tagging = ASN1EXPL, 
                      int length = 0);

   /**
    * This method decodes a variable of the ASN.1 REAL type.
    *
    * @param val                       Reference to a variable to receive the
    *                                    decoded REAL value.
    * @param tagging                   An enumerated type whose value is set to
    *                                    either 'ASN1EXPL' (for explicit
    *                                    tagging) or 'ASN1IMPL' (for implicit).
    *                                    Controls whether the universal tag
    *                                    value for this type is added or not.
    *                                    Users will generally always set this
    *                                    value to 'ASN1EXPL'.
    * @param length                    The length, in octets, of the contents
    *                                    field to be decoded. This parameter
    *                                    only has meaning if the tagging
    *                                    parameter specifies implicit decoding.
    *                                    If explicit, the length is obtained
    *                                    from the decoded length field.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmReal
    */
   void decodeReal (ASN1REAL& val, ASN1TagType tagging = ASN1EXPL, 
                    int length = 0);

   /**
    * This method decodes a variable of the ASN.1 RELATIVE-OID type.
    *
    * @param val                       Reference to a variable to receive the
    *                                    decoded RELATIVE-OID value.
    * @param tagging                   An enumerated type whose value is set to
    *                                    either 'ASN1EXPL' (for explicit
    *                                    tagging) or 'ASN1IMPL' (for implicit).
    *                                    Controls whether the universal tag
    *                                    value for this type is added or not.
    *                                    Users will generally always set this
    *                                    value to 'ASN1EXPL'.
    * @param length                    The length, in octets, of the contents
    *                                    field to be decoded. This parameter
    *                                    only has meaning if the tagging
    *                                    parameter specifies implicit decoding.
    *                                    If explicit, the length is obtained
    *                                    from the decoded length field.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmRelativeOID
    */
   void decodeRelativeOID (ASN1OBJID& val, ASN1TagType tagging = ASN1EXPL, 
                           int length = 0);

   /**
    * This method decodes the tag at the current decode pointer location and
    * returns the results.
    *
    * @param tag                       Reference to a variable to receive the
    *                                    decoded ASN.1 tag value.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmTag
    */
   void decodeTag (ASN1TAG& tag);

   /**
    * This method decodes the tag and length at the current decode pointer
    * location and returns the results.
    *
    * @param tag                       Reference to a variable to receive the
    *                                    decoded ASN.1 tag value.
    * @param len                       Reference to a variable to receive the
    *                                    decoded length of the tagged
    *                                    component. The returned value will
    *                                    either be the actual length or the
    *                                    special constant 'ASN_K_INDEFLEN',
    *                                    which indicates indefinite length.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmTagAndLen
    */
   void decodeTagAndLen (ASN1TAG& tag, ASN1INT& len);

   /**
    * This method decodes a variable of the unsigned variant of ASN.1 INTEGER
    * type.
    *
    * @param val                       Reference to a variable to receive a
    *                                    decoded unsigned 32-bit integer value.
    * @param tagging                   An enumerated type whose value is set to
    *                                    either 'ASN1EXPL' (for explicit
    *                                    tagging) or 'ASN1IMPL' (for implicit).
    *                                    Controls whether the universal tag
    *                                    value for this type is added or not.
    *                                    Users will generally always set this
    *                                    value to 'ASN1EXPL'.
    * @param length                    The length, in octets, of the contents
    *                                    field to be decoded. This parameter
    *                                    only has meaning if the tagging
    *                                    parameter specifies implicit decoding.
    *                                    If explicit, the length is obtained
    *                                    from the decoded length field.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmUInt
    */
   void decodeUInt (ASN1UINT& val, ASN1TagType tagging = ASN1EXPL, 
                    int length = 0);

   /**
    * This method decodes an 8-bit variable of the unsigned variant of ASN.1
    * INTEGER type.
    *
    * @param val                       Reference to a variable to receive a
    *                                    decoded unsigned 8-bit integer value.
    * @param tagging                   An enumerated type whose value is set to
    *                                    either 'ASN1EXPL' (for explicit
    *                                    tagging) or 'ASN1IMPL' (for implicit).
    *                                    Controls whether the universal tag
    *                                    value for this type is added or not.
    *                                    Users will generally always set this
    *                                    value to 'ASN1EXPL'.
    * @param length                    The length, in octets, of the contents
    *                                    field to be decoded. This parameter
    *                                    only has meaning if the tagging
    *                                    parameter specifies implicit decoding.
    *                                    If explicit, the length is obtained
    *                                    from the decoded length field.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmUInt8
    */
   void decodeUInt8 (ASN1UINT8& val, ASN1TagType tagging = ASN1EXPL, 
                     int length = 0);

   /**
    * This method decodes a 16-bit variable of the unsigned variant of ASN.1
    * INTEGER type.
    *
    * @param val                       Reference to a variable to receive a
    *                                    decoded unsigned 16-bit integer value.
    * @param tagging                   An enumerated type whose value is set to
    *                                    either 'ASN1EXPL' (for explicit
    *                                    tagging) or 'ASN1IMPL' (for implicit).
    *                                    Controls whether the universal tag
    *                                    value for this type is added or not.
    *                                    Users will generally always set this
    *                                    value to 'ASN1EXPL'.
    * @param length                    The length, in octets, of the contents
    *                                    field to be decoded. This parameter
    *                                    only has meaning if the tagging
    *                                    parameter specifies implicit decoding.
    *                                    If explicit, the length is obtained
    *                                    from the decoded length field.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmUInt16
    */
   void decodeUInt16 (ASN1USINT& val, ASN1TagType tagging = ASN1EXPL, 
                      int length = 0);

   /**
    * This method decodes a 64-bit variable of the unsigned variant of ASN.1
    * INTEGER type.
    *
    * @param val                       Reference to a variable to receive a
    *                                    decoded unsigned 64-bit integer value.
    * @param tagging                   An enumerated type whose value is set to
    *                                    either 'ASN1EXPL' (for explicit
    *                                    tagging) or 'ASN1IMPL' (for implicit).
    *                                    Controls whether the universal tag
    *                                    value for this type is added or not.
    *                                    Users will generally always set this
    *                                    value to 'ASN1EXPL'.
    * @param length                    The length, in octets, of the contents
    *                                    field to be decoded. This parameter
    *                                    only has meaning if the tagging
    *                                    parameter specifies implicit decoding.
    *                                    If explicit, the length is obtained
    *                                    from the decoded length field.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmUInt64
    */
   void decodeUInt64 (ASN1UINT64& val, ASN1TagType tagging = ASN1EXPL, 
                      int length = 0);

   /**
    * This method decodes a variable of the ASN.1 UniversalString type.
    *
    * @param val                       Reference to a variable to receive the
    *                                    decoded UniversalString value.
    * @param tagging                   An enumerated type whose value is set to
    *                                    either 'ASN1EXPL' (for explicit
    *                                    tagging) or 'ASN1IMPL' (for implicit).
    *                                    Controls whether the universal tag
    *                                    value for this type is added or not.
    *                                    Users will generally always set this
    *                                    value to 'ASN1EXPL'.
    * @param length                    The length, in octets, of the contents
    *                                    field to be decoded. This parameter
    *                                    only has meaning if the tagging
    *                                    parameter specifies implicit decoding.
    *                                    If explicit, the length is obtained
    *                                    from the decoded length field.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::berDecStrmUnivStr
    */
   void decodeUnivStr (Asn132BitCharString& val, ASN1TagType tagging = ASN1EXPL, 
                       int length = 0);

   /**
    * This method marks the current position in this input stream. A subsequent
    * call to the ASN1BERInputStream::reset function repositions this stream at
    * the last marked position so that subsequent reads re-read the same bytes.
    * The readAheadLimit argument tells this input stream to allow that many
    * bytes to be read before the mark position gets invalidated.
    *
    * @param readAheadLimit            the maximum limit of bytes that can be
    *                                    read before the mark position becomes
    *                                    invalid.
    * @exception OSCStreamException    if error occurred.
    * @see                            ::rtStreamBufMark, ::rtStreamBufReset
    */
   void mark (int readAheadLimit);

   /**
    * Repositions this stream to the position at the time the mark method was
    * last called on this input stream.
    *
    * @exception OSCStreamException    if error occurred.
    * @see                            ::rtStreamBufMark, ::rtStreamBufReset
    */
   void reset ();
} ;

/**
 * A file input stream for streaming BER decoding. This class reads data from a
 * file.
 */
class EXTERNBER ASN1BERFileInputStream : public ASN1BERInputStream {
 public:
   /**
    * Creates and initializes the file input stream using the name of file.
    *
    * @param pFilename                 Name of file.
    * @exception OSCStreamException    stream can't be created or initialized.
    * @see                            ::rtStreamFileOpen
    */
   ASN1BERFileInputStream (const char* pFilename);

   /**
    * Initializes the file input stream using the opened FILE structure
    * descriptor.
    *
    * @param file                      Pointer to FILE structure.
    * @exception OSCStreamException    stream can't be created or initialized.
    * @see                            ::rtStreamFileAttach
    */
   ASN1BERFileInputStream (FILE* file);
} ;

/**
 * A socket input stream for streaming BER decoding. This class reads data
 * directly from a network via TCP/IP socket.
 */
class EXTERNBER ASN1BERSocketInputStream : public ASN1BERInputStream {
 protected:
   OSCSocket mSocket; ///< a socket 
 public:
   /**
    * Creates and initializes the socket input stream using the socket's
    * handle.
    *
    * @param socket                    Handle of the socket.
    * @param ownership                 Indicates the ownership of the socket's
    *                                    handle.
    * @exception OSCStreamException    stream can't be created or initialized.
    * @see                            ::rtSocketCreate, ::rtSocketAccept
    * @see                            ::rtStreamSocketAttach
    */
   ASN1BERSocketInputStream (OSRTSOCKET socket, ASN1BOOL ownership = FALSE);

   /**
    * Initializes the socket input stream using the OSCSocket class' instance.
    *
    * @param socket                    Reference to OSCSocket class' instance.
    * @exception OSCStreamException    stream can't be created or initialized.
    */
   ASN1BERSocketInputStream (OSCSocket& socket);
} ;

/** @} */

#endif /* _OSCINPUTCPPSTREAM_H_ */

